<?php

use \Recurr\Rule;

// Exit if accessed directly
if (! defined('ABSPATH')) {
	exit;
}

// Check if class already exists
if (! class_exists('acf_field_rrule')) :

	class acf_field_rrule extends acf_field {

		/*
		 *  __construct
		 *
		 *  This function will setup the field type data
		 *
		 *  @type	function
		 *  @date	5/03/2014
		 *  @since	5.0.0
		 *
		 *  @param	n/a
		 *  @return	n/a
		 */

		function __construct( $settings ) {

			/*
			 *  name (string) Single word, no spaces. Underscores allowed
			 */

			$this->name = 'rrule';

			/*
			 *  label (string) Multiple words, can include spaces, visible when selecting a field type
			 */

			$this->label = __('RRule', 'acf-rrule');

			/*
			 *  category (string) basic | content | choice | relational | jquery | layout | CUSTOM GROUP NAME
			 */

			$this->category = 'jquery';

			/*
			 *  defaults (array) Array of default settings which are merged into the field object. These are used later in settings
			 */

			$this->defaults = array(
				'date_display_format' => 'j F Y',
				'date_return_format' => 'Y-m-d',
				'allow_time' => false,
				'time_display_format' => 'H:i',
				'timezone' => get_option('timezone_string'),
			);

			/*
			 *  l10n (array) Array of strings that are used in JavaScript. This allows JS strings to be translated in PHP and loaded via:
			 *  var message = acf._e('rrule', 'error');
			 */

			// $this->l10n = array(
			// 	'error'	=> __('Error! Please enter a higher value', 'acf-rrule'),
			// );

			/*
			 *  settings (array) Store plugin settings (url, path, version) as a reference for later use with assets
			 */

			$this->settings = $settings;

			parent::__construct();

		}

		/*
	 	 *  render_field_settings()
		 *
		 *  Create extra settings for your field. These are visible when editing a field
		 *
		 *  @type	action
		 *  @since	3.6
		 *  @date	23/01/13
		 *
		 *  @param	$field (array) the $field being edited
		 *  @return	n/a
		 */

		function render_field_settings( $field ) {

			/*
			 *  acf_render_field_setting
			 *
			 *  This function will create a setting for your field. Simply pass the $field parameter and an array of field settings.
			 *  The array of settings does not require a `value` or `prefix`; These settings are found from the $field array.
			 *
			 *  More than one setting can be added by copy/paste the above code.
			 *  Please note that you must also have a matching $defaults value for the field name (font_size)
			 */

			global $wp_locale;

			// Init vars
			$d_m_Y = date_i18n('d/m/Y');
			$m_d_Y = date_i18n('m/d/Y');
			$F_j_Y = date_i18n('F j, Y');
			$Ymd = date_i18n('Ymd');

			// Display format
			acf_render_field_setting( $field, array(
				'label'			=> __('Date Display Format', 'acf-rrule'),
				'instructions'	=> __('The date format displayed when editing a post', 'acf-rrule'),
				'type'			=> 'radio',
				'name'			=> 'date_display_format',
				'other_choice'	=> 1,
				'choices'		=> array(
					'd/m/Y'			=> '<span>' . $d_m_Y . '</span><code>d/m/Y</code>',
					'm/d/Y'			=> '<span>' . $m_d_Y . '</span><code>m/d/Y</code>',
					'F j, Y'		=> '<span>' . $F_j_Y . '</span><code>F j, Y</code>',
					'other'			=> '<span>' . __('Custom:', 'acf') . '</span>'
				)
			));

			// Return format
			acf_render_field_setting( $field, array(
				'label'			=> __('Date Return Format', 'acf-rrule'),
				'instructions'	=> __('The date format returned via template functions', 'acf-rrule'),
				'type'			=> 'radio',
				'name'			=> 'return_format',
				'other_choice'	=> 1,
				'choices'		=> array(
					'd/m/Y'			=> '<span>' . $d_m_Y . '</span><code>d/m/Y</code>',
					'm/d/Y'			=> '<span>' . $m_d_Y . '</span><code>m/d/Y</code>',
					'F j, Y'		=> '<span>' . $F_j_Y . '</span><code>F j, Y</code>',
					'Ymd'			=> '<span>' . $Ymd . '</span><code>Ymd</code>',
					'other'			=> '<span>' . __('Custom:', 'acf') . '</span>'
				)
			));

			// Allow time selector
			acf_render_field_setting( $field, array(
				'label'			=> __('Time Selector', 'acf-rrule'),
				'instructions'	=> __('Allow time selection when creating the recurring rule', 'acf-rrule'),
				'name'			=> 'allow_time',
				'type'			=> 'true_false',
				'ui'			=> 1,
			));

		}


		/*
		 *  render_field()
		 *
		 *  Create the HTML interface for your field
		 *
		 *  @param	$field (array) the $field being rendered
		 *
		 *  @type	action
		 *  @since	3.6
		 *  @date	23/01/13
		 *
		 *  @param	$field (array) the $field being edited
		 *  @return	n/a
		 */

		function render_field( $field ) {

			// Generate a unique ID for fields we don't want to be autocompleted
			$unique_id = $field['id'] . '-' . time();

			// Datepicker options
			$datepicker_options = array(
				'class' => 'acf-date-picker acf-input-wrap',
				'data-date_format' => acf_convert_date_to_js($field['date_display_format']),
			);
			$timepicker_options = array(
				'class' => 'acf-time-picker acf-input-wrap',
			);

			// HTML
			?>

	        <?php if ($field['value'] && $field['value']['text']) : ?>
	            <p class="acf-field-rrule-current">
	                <?=__('Current value:', 'acf-rrule')?> <?=$field['value']['text']?>
	            </p>
	        <?php endif; ?>

			<div class="acf-field-rrule-sub-fields">
				<div class="acf-field">
					<div class="acf-columns">
						<div class="acf-column">
							<div class="acf-field acf-field-date-picker is-required" data-type="date_picker">
								<div <?php acf_esc_attr_e( $datepicker_options ); ?>>
									<?php
									$start_date_hidden = '';
									$start_date_display = '';

									// Format values
									if ($field['value']) {
										$start_date_hidden = acf_format_date( $field['value']['start_date'], 'Ymd' );
										$start_date_display = acf_format_date( $field['value']['start_date'], $field['date_display_format'] );
									}
									?>

									<div class="acf-label">
										<label for="<?=$unique_id?>-start-date">
											<?=__('Start date', 'acf-rrule')?> <span class="acf-required">*</span>
										</label>
									</div>

									<?php acf_hidden_input( array (
										'name' => $field['name'] . '[start_date]',
										'value'	=> $start_date_hidden,
									) ); ?>
									<?php acf_text_input( array(
										'id' => $unique_id . '-start-date',
										'class' => 'input',
										'value'	=> $start_date_display,
									) ); ?>
								</div>
							</div>
						</div>

						<?php if ($field['allow_time']) : ?>
							<div class="acf-column">
								<div class="acf-field acf-field-time-picker is-required" data-type="time_picker">
									<div <?php acf_esc_attr_e( $timepicker_options ); ?>>
										<?php
										$start_time = '';

										// Format values
										if ($field['value']) {
											$start_time = acf_format_date( $field['value']['start_time'], $field['time_display_format'] );
										}
										?>

										<div class="acf-label">
											<label for="<?=$unique_id?>-start-time">
												<?=__('Start time', 'acf-rrule')?> <span class="acf-required">*</span>
											</label>
										</div>

										<?php acf_hidden_input( array (
											'name' => $field['name'] . '[start_time]',
											'value'	=> $start_time,
										) ); ?>
										<?php acf_text_input( array(
											'id' => $unique_id . '-start-time',
											'class' => 'input',
											'value'	=> $start_time,
										) ); ?>
									</div>
								</div>
							</div>
						<?php endif; ?>
					</div>
				</div>

				<div class="acf-field">
					<?php
					$frequency = array(
						'id' => $field['id'] . '-frequency',
						'name' => $field['name'] . '[frequency]',
						'value' => is_array($field['value']) ? $field['value']['frequency'] : null,
	                    'class' => 'frequency-select',
						'choices' => array(
							'DAILY' => __('Daily', 'acf-rrule'),
							'WEEKLY' => __('Weekly', 'acf-rrule'),
							'MONTHLY' => __('Monthly', 'acf-rrule'),
							'YEARLY' => __('Yearly', 'acf-rrule'),
						),
					);
					?>

					<div class="acf-label">
						<label for="<?=$frequency['id']?>">
							<?=__('Frequency', 'acf-rrule')?>
						</label>
					</div>

					<div class="acf-input">
						<?php acf_select_input( $frequency ); ?>
					</div>
				</div>

				<div class="acf-field">
					<?php
					$interval = array(
						'id' => $field['id'] . '-interval',
						'name' => $field['name'] . '[interval]',
						'type' => 'number',
						'class' => 'acf-is-prepended acf-is-appended',
						'value'	=> is_array($field['value']) && $field['value']['interval'] ? $field['value']['interval'] : 1,
						'min' => 1,
						'step' => 1,
					);
					?>

					<div class="acf-input">
						<div class="acf-input-prepend"><?=_x('Every', 'RRule Interval', 'acf-rrule')?></div>
						<div class="acf-input-append">
							<span class="freq-suffix" data-frequency="DAILY"><?=_x('day', 'RRule Interval', 'acf-rrule')?></span>
							<span class="freq-suffix" data-frequency="WEEKLY"><?=_x('week', 'RRule Interval', 'acf-rrule')?></span>
							<span class="freq-suffix" data-frequency="MONTHLY"><?=_x('month', 'RRule Interval', 'acf-rrule')?></span>
							<span class="freq-suffix" data-frequency="YEARLY"><?=_x('year', 'RRule Interval', 'acf-rrule')?></span>
						</div>
						<div class="acf-input-wrap">
							<?php acf_text_input( $interval ); ?>
						</div>
					</div>
				</div>

				<div class="acf-field acf-field-button-group" data-type="button_group_multiple" data-frequency="WEEKLY">
					<?php
					$weekdays = array(
						'MO' => __('Monday', 'acf-rrule'),
						'TU' => __('Tuesday', 'acf-rrule'),
						'WE' => __('Wednesday', 'acf-rrule'),
						'TH' => __('Thursday', 'acf-rrule'),
						'FR' => __('Friday', 'acf-rrule'),
						'SA' => __('Saturday', 'acf-rrule'),
						'SU' => __('Sunday', 'acf-rrule'),
					);
					?>

					<div class="acf-label">
						<label>
							<?=__('Week days', 'acf-rrule')?>
						</label>
					</div>

					<div class="acf-input">
						<div class="acf-button-group">
							<?php foreach ($weekdays as $key => $value) : ?>
	                            <?php $selected = is_array($field['value']) && in_array($key, $field['value']['weekdays']); ?>

								<label<?=($selected ? ' class="selected"' : '')?>>
									<input type="checkbox" name="<?=$field['name']?>[weekdays][]" value="<?=$key?>"<?=($selected ? ' checked' : '')?>> <?=$value?>
								</label>
							<?php endforeach; ?>
						</div>
					</div>
				</div>

				<div id="<?=$field['id']?>-monthly-by" class="acf-field monthly-by-options" data-frequency="MONTHLY">
					<div class="acf-columns">
						<div class="acf-column">
							<div class="acf-label is-inline">
								<input id="acf-<?=$field['name']?>-bymonthdays" type="radio" name="<?=$field['name']?>[monthly_by]" value="monthdays"<?=(is_array($field['value']) && $field['value']['monthly_by'] == 'monthdays' ? ' checked' : '')?>>
								<label for="acf-<?=$field['name']?>-bymonthdays">
									<?=__('Month days', 'acf-rrule')?>
								</label>
							</div>

							<?php $days = range(1, 31); ?>

							<div class="acf-input<?=$field['value']['monthly_by'] != 'monthdays' ? ' is-disabled' : ''?>" data-monthly-by="monthdays">
								<table class="acf-rrule-monthdays">
									<?php foreach (array_chunk($days, 7) as $week) : ?>
										<tr>
											<?php foreach ($week as $day) : ?>
	                                            <?php $selected = is_array($field['value']) && in_array($day, $field['value']['monthdays']); ?>

												<td>
													<input id="acf-<?=$field['name']?>-monthdays-<?=$day?>" type="checkbox" name="<?=$field['name']?>[monthdays][]" value="<?=$day?>"<?=($selected ? ' checked' : '')?>>
													<label for="acf-<?=$field['name']?>-monthdays-<?=$day?>"><?=$day?></label>
												</td>
											<?php endforeach; ?>
										</tr>
									<?php endforeach; ?>
								</table>
							</div>
						</div>

						<div class="acf-column">
							<div class="acf-label is-inline">
								<input id="acf-<?=$field['name']?>-bysetpos" type="radio" name="<?=$field['name']?>[monthly_by]" value="setpos"<?=(is_array($field['value']) && $field['value']['monthly_by'] == 'setpos' ? ' checked' : '')?>>
								<label for="acf-<?=$field['name']?>-bysetpos">
									<?=__('Day of the week', 'acf-rrule')?>
								</label>
							</div>

							<?php
							$setpos = array(
								'id' => $field['id'] . '-setpos',
								'name' => $field['name'] . '[setpos]',
								'value' => is_array($field['value']) ? $field['value']['setpos'] : null,
								'choices' => array(
									'1' => __('First', 'acf-rrule'),
									'2' => __('Second', 'acf-rrule'),
									'3' => __('Third', 'acf-rrule'),
									'4' => __('Fourth', 'acf-rrule'),
									'-1' => __('Last', 'acf-rrule'),
								),
							);
							$setpos_options = array(
								'id' => $field['id'] . '-setpos-option',
								'name' => $field['name'] . '[setpos_option]',
								'value' => is_array($field['value']) ? $field['value']['setpos_option'] : null,
								'choices' => $weekdays,
							);
							?>

							<div class="acf-input<?=(is_array($field['value']) && $field['value']['monthly_by'] != 'setpos' ? ' is-disabled' : '')?>" data-monthly-by="setpos">
								<div class="acf-columns">
									<div class="acf-column">
										<?php acf_select_input( $setpos ); ?>
									</div>
									<div class="acf-column">
										<?php acf_select_input( $setpos_options ); ?>
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>

				<div class="acf-field acf-field-checkbox" data-type="checkbox" data-key="<?=$field['key']?>" data-frequency="YEARLY">
					<div class="acf-label">
						<label><?=__('Month', 'acf-rrule')?></label>
					</div>
					<div class="acf-input">
						<?php $months = array(
							'1' => __('January', 'acf-rrule'),
							'2' => __('February', 'acf-rrule'),
							'3' => __('March', 'acf-rrule'),
							'4' => __('April', 'acf-rrule'),
							'5' => __('May', 'acf-rrule'),
							'6' => __('June', 'acf-rrule'),
							'7' => __('July', 'acf-rrule'),
							'8' => __('August', 'acf-rrule'),
							'9' => __('September', 'acf-rrule'),
							'10' => __('October', 'acf-rrule'),
							'11' => __('November', 'acf-rrule'),
							'12' => __('December', 'acf-rrule'),
						); ?>

						<input type="hidden" name="<?=$field['name']?>[months]">

						<ul class="acf-checkbox-list acf-hl">
							<?php foreach ($months as $key => $month) : ?>
	                            <?php $selected = is_array($field['value']) && in_array($key, $field['value']['months']); ?>

								<li>
									<label<?=($selected ? ' class="selected"' : '')?>>
										<input type="checkbox" id="acf-<?=$field['name']?>-months-<?=$key?>" name="<?=$field['name']?>[months][]" value="<?=$key?>"<?=($selected ? ' checked' : '')?>>
										<?=$month?>
									</label>
								</li>
							<?php endforeach; ?>
						</ul>
					</div>
				</div>

				<div class="acf-field">
					<div class="acf-label">
						<label for="<?=$field['id']?>-end-type">
							<?=__('End date', 'acf-rrule')?>
						</label>
					</div>
					<div class="acf-input">
						<?php acf_select_input( array(
							'id' => $field['id'] . '-end-type',
							'name' => $field['name'] . '[end_type]',
							'value' => $field['value'] ? $field['value']['end_type'] : null,
	                        'class' => 'end-type-select',
							'choices' => array(
								'date' => __('At a specific date', 'acf-rrule'),
								'count' =>  __('After a number of occurences', 'acf-rrule'),
							),
						) ); ?>
					</div>
				</div>

				<div class="acf-field" data-end-type="count">
					<?php
					$occurence_count = array(
						'id' => $field['id'] . '-occurence-count',
						'name' => $field['name'] . '[occurence_count]',
						'type' => 'number',
						'class' => 'acf-is-prepended acf-is-appended',
						'value'	=> is_array($field['value']) ? $field['value']['occurence_count'] : null,
						'min' => 1,
						'step' => 1,
					);
					?>

					<div class="acf-input">
						<div class="acf-input-prepend"><?=__('After', 'acf-rrule')?></div>
						<div class="acf-input-append"><?=__('occurence(s)', 'acf-rrule')?></div>
						<div class="acf-input-wrap">
							<?php acf_text_input( $occurence_count ); ?>
						</div>
					</div>
				</div>

				<div class="acf-field acf-field-date-picker" data-type="date_picker" data-end-type="date">
					<div <?php acf_esc_attr_e( $datepicker_options ); ?>>
						<?php
						$end_date_hidden = '';
						$end_date_display = '';

						// Format values
						if ($field['value']) {
							$end_date_hidden = acf_format_date( $field['value']['end_date'], 'Ymd' );
							$end_date_display = acf_format_date( $field['value']['end_date'], $field['date_display_format'] );
						}
						?>

						<div class="acf-input">
							<div class="acf-input-prepend"><?=__('Until', 'acf-rrule')?></div>
							<div class="acf-input-wrap">
								<?php acf_hidden_input( array (
									'name' => $field['name'] . '[end_date]',
									'value'	=> $end_date_hidden,
								) ); ?>
								<?php acf_text_input( array(
									'id' => $unique_id . '-end-date',
									'class' => 'acf-is-prepended',
									'value'	=> $end_date_display,
								) ); ?>
							</div>
						</div>
					</div>
				</div>
			</div>

			<?php
		}

		/*
		 *  input_admin_enqueue_scripts()
		 *
		 *  This action is called in the admin_enqueue_scripts action on the edit screen where your field is created.
		 *  Use this action to add CSS + JavaScript to assist your render_field() action.
		 *
		 *  @type	action (admin_enqueue_scripts)
		 *  @since	3.6
		 *  @date	23/01/13
		 *
		 *  @param	n/a
		 *  @return	n/a
		 */

		function input_admin_enqueue_scripts() {

			// Bail early if no enqueue
		   	if (! acf_get_setting('enqueue_datepicker')) {
			   	return;
		   	}

		   	// Localize data
		   	global $wp_locale;

		   	acf_localize_data(array(
			   	'datePickerL10n'	=> array(
					'closeText'			=> _x('Done',	'Date Picker JS closeText',		'acf'),
					'currentText'		=> _x('Today',	'Date Picker JS currentText',	'acf'),
					'nextText'			=> _x('Next',	'Date Picker JS nextText',		'acf'),
					'prevText'			=> _x('Prev',	'Date Picker JS prevText',		'acf'),
					'weekHeader'		=> _x('Wk',		'Date Picker JS weekHeader',	'acf'),
					'monthNames'        => array_values( $wp_locale->month ),
					'monthNamesShort'   => array_values( $wp_locale->month_abbrev ),
					'dayNames'          => array_values( $wp_locale->weekday ),
					'dayNamesMin'       => array_values( $wp_locale->weekday_initial ),
					'dayNamesShort'     => array_values( $wp_locale->weekday_abbrev )
				)
		   	));

			// Enqueue scripts
			wp_enqueue_script('jquery-ui-datepicker');

			// Enqueue style
			wp_enqueue_style('acf-datepicker', acf_get_url('assets/inc/datepicker/jquery-ui.min.css'), array(), '1.11.4' );

			// Init vars
			$url = $this->settings['url'];
			$version = $this->settings['version'];

			// Register & include JS
			wp_enqueue_script('acf-rrule', "{$url}assets/js/input.js", array('acf-input'), $version);

			// Register & include CSS
			wp_enqueue_style('acf-rrule', "{$url}assets/css/input.css", array('acf-input'), $version);

		}

		/*
		 *  load_value()
		 *
		 *  This filter is applied to the $value after it is loaded from the db
		 *
		 *  @type	filter
		 *  @since	3.6
		 *  @date	23/01/13
		 *
		 *  @param	$value (mixed) the value found in the database
		 *  @param	$post_id (mixed) the $post_id from which the value was loaded
		 *  @param	$field (array) the field array holding all the field options
		 *  @return	$value
		 */

		function load_value( $value, $post_id, $field ) {

			$new_value = array(
				'rrule' => null,
				'start_date' => null,
				'start_time' => null,
				'frequency' => 'WEEKLY',
				'interval' => 1,
				'weekdays' => array(),
				'monthdays' => array(),
				'months' => array(),
				'monthly_by' => 'monthdays',
				'setpos' => 1,
				'setpos_option' => 'MO',
				'end_type' => null,
				'end_date' => null,
				'occurence_count' => null,
	            'dates_collection' => null,
	            'text' => null,
			);

			if ($value) {
				try {
					$rule = new Rule($value);

					$start_date = $rule->getStartDate();

					$new_value['rrule'] = $value;
					$new_value['start_date'] = $start_date->format('Ymd');
					$new_value['start_time'] = $start_date->format('H:i:s');
					$new_value['frequency'] = $rule->getFreqAsText();
					$new_value['interval'] = $rule->getInterval();
					$new_value['weekdays'] = $rule->getByDay() ?: array();
					$new_value['monthdays'] = $rule->getByMonthDay() ?: array();
					$new_value['months'] = $rule->getByMonth() ?: array();

					if ($new_value['frequency'] == 'MONTHLY') {
						if (sizeof($new_value['weekdays']) > 0) {
							$new_value['monthly_by'] = 'setpos';
							$set_position = $rule->getBySetPosition();
							$new_value['setpos'] = $set_position[0];
							$new_value['setpos_option'] = $new_value['weekdays'][0];
						} else {
							$new_value['monthly_by'] = 'monthdays';
						}
					}

					if ($rule->getUntil()) {
						$new_value['end_type'] = 'date';
						$new_value['end_date'] =  $rule->getUntil()->format('Ymd');
					} else {
						$new_value['end_type'] = 'count';
						$new_value['occurence_count'] =  $rule->getCount();
					}

	                $locale = explode('_', get_locale());

	                $transformer = new \Recurr\Transformer\ArrayTransformer();
	                $textTransformer = new \Recurr\Transformer\TextTransformer(
	                    new \Recurr\Transformer\Translator($locale[0])
	                );

	                $new_value['dates_collection'] = array();

	                foreach ($transformer->transform($rule) as $recurrence) {
	                    $new_value['dates_collection'][] = $recurrence->getStart();
	                }

	                $new_value['text'] = $textTransformer->transform($rule);
				} catch (\Exception $e) {
					//
				}
			} else {
	            $new_value = false;
	        }

			return $new_value;

		}

		/*
		 *  update_value()
		 *
		 *  This filter is applied to the $value before it is saved in the db
		 *
		 *  @type	filter
		 *  @since	3.6
		 *  @date	23/01/13
		 *
		 *  @param	$value (mixed) the value found in the database
		 *  @param	$post_id (mixed) the $post_id from which the value was loaded
		 *  @param	$field (array) the field array holding all the field options
		 *  @return	$value
		 */

		function update_value( $value, $post_id, $field ) {

			if (is_array($value)) {
				$start_date = \DateTime::createFromFormat('Ymd', $value['start_date']);

				// Bail early if the DateTime object is null
				if (! $start_date) {
					return $value;
				}

				if (array_key_exists('start_time', $value)) {
					$start_time = array_pad(explode(':', $value['start_time']), 3, 0);
					$start_date->setTime(intval($start_time[0]), intval($start_time[1]), intval($start_time[2]));
				} else {
					$start_date->setTime(0,0,0);
				}

				$rule = new Rule;

				$rule->setTimezone($field['timezone'])
					 ->setStartDate($start_date, true)
					 ->setFreq($value['frequency'])
					 ->setInterval($value['interval']);

				switch ($value['frequency']) {
					case 'WEEKLY':
						$rule->setByDay($value['weekdays']);

						break;

					case 'MONTHLY':
						if ($value['monthly_by'] == 'monthdays') {
							$rule->setByMonthDay($value['monthdays']);
						} else {
							$rule->setBySetPosition(array(intval($value['setpos'])));
							$rule->setByDay(array($value['setpos_option']));
						}

						break;

					case 'YEARLY':
						$rule->setByMonth($value['months']);

						break;

					default: break;
				}

				switch ($value['end_type']) {
					case 'date':
						if ($value['end_date']) {
							$end_date = \DateTime::createFromFormat('Ymd', $value['end_date']);
							$end_date->setTime(0,0,0);

							$rule->setUntil($end_date);
						}

						break;
					case 'count':
						$rule->setCount($value['occurence_count']);
						break;
					default: break;
				}

				$new_value = $rule->getString();

				acf_update_value( $new_value, $post_id, $field );

				return $new_value;
			}

	        return $value;
		}

	}

	// Initialize RRule
	new acf_field_rrule( $this->settings );

// class_exists check
endif;

?>
